from __future__ import annotations

import subprocess
from typing import Dict, Literal, Optional


PlanPreset = Literal["balanced", "power_saver", "high_performance"]


def _run_powercfg(arguments: list[str]) -> subprocess.CompletedProcess[str]:
    return subprocess.run(
        ["powercfg", *arguments],
        capture_output=True,
        text=True,
        check=False,
    )


def list_power_plans() -> Dict[str, str]:
    """
    Returns a mapping of power plan GUID to friendly name.
    """
    completed = _run_powercfg(["/L"])
    plans: Dict[str, str] = {}
    if completed.returncode != 0:
        return plans

    for line in completed.stdout.splitlines():
        line = line.strip()
        if not line or "GUID" not in line:
            continue
        parts = line.split(":")
        if len(parts) < 3:
            continue
        guid_part = parts[1].strip()
        name_part = parts[2].strip()
        guid = guid_part.split(" ")[0]
        name = name_part.replace("*", "").strip()
        plans[guid] = name
    return plans


def set_active_plan(plan_guid: str) -> str:
    completed = _run_powercfg(["/S", plan_guid])
    if completed.returncode != 0:
        raise RuntimeError(completed.stderr.strip() or f"Failed to set plan {plan_guid}")
    return completed.stdout.strip()


def duplicate_power_plan(preset: PlanPreset, name: str) -> str:
    preset_map: Dict[PlanPreset, str] = {
        "balanced": "381b4222-f694-41f0-9685-ff5bb260df2e",
        "power_saver": "a1841308-3541-4fab-bc81-f71556f20b4a",
        "high_performance": "8c5e7fda-e8bf-4a96-9a85-a6e23a8c635c",
    }
    completed = _run_powercfg(["/DUPLICATESCHEME", preset_map[preset]])
    if completed.returncode != 0:
        raise RuntimeError(completed.stderr.strip() or "Failed to duplicate plan")

    for line in completed.stdout.splitlines():
        line = line.strip()
        if "GUID" in line:
            guid = line.split(":")[1].strip()
            rename = _run_powercfg(["/CHANGENAME", guid, name])
            if rename.returncode != 0:
                raise RuntimeError(rename.stderr.strip() or "Failed to rename plan")
            return guid

    raise RuntimeError("Could not extract duplicated plan GUID")


def configure_power_settings(plan_guid: str) -> None:
    """
    Apply conservative settings for the given plan (reduce CPU boost, shorter display timeout).
    """
    settings = [
        ["/SETACVALUEINDEX", plan_guid, "SUB_PROCESSOR", "PROCTHROTTLEMIN", "10"],
        ["/SETACVALUEINDEX", plan_guid, "SUB_PROCESSOR", "PROCTHROTTLEMAX", "60"],
        ["/SETDCVALUEINDEX", plan_guid, "SUB_PROCESSOR", "PROCTHROTTLEMIN", "5"],
        ["/SETDCVALUEINDEX", plan_guid, "SUB_PROCESSOR", "PROCTHROTTLEMAX", "50"],
        ["/SETACVALUEINDEX", plan_guid, "SUB_VIDEO", "VIDEOIDLE", "180"],
        ["/SETDCVALUEINDEX", plan_guid, "SUB_VIDEO", "VIDEOIDLE", "60"],
    ]

    for args in settings:
        completed = _run_powercfg(args)
        if completed.returncode != 0:
            raise RuntimeError(completed.stderr.strip() or f"Failed to apply {args}")


def disable_hibernation() -> Optional[str]:
    completed = _run_powercfg(["/hibernate", "off"])
    if completed.returncode != 0:
        return completed.stderr.strip()
    return completed.stdout.strip()

